// middlewares/moderationLogger.js
const ModerationLog = require("../model/ModerationLog");

function moderationLogger(action, targetType) {
  return (req, res, next) => {
    // Store original res.json to capture response data if needed
    const originalJson = res.json;

    res.on("finish", async () => {
      try {
        // Only log successful operations
        if (res.statusCode >= 200 && res.statusCode < 300) {
          // ✅ Check req.user HERE, not earlier
          if (!req.user?.id) {
            console.warn(
              "⚠️ moderationLogger: req.user not set, skipping log."
            );
            return;
          }

          const targetId =
            req.params.id || req.params[`${targetType}Id`] || req.body.targetId;

          if (!targetId) {
            console.warn(
              `⚠️ moderationLogger: Missing targetId for ${action} on ${targetType}.`
            );
            return;
          }

          await ModerationLog.create({
            action,
            targetType,
            targetId,
            performedBy: req.user.id, // Use id, not _id
            reason: req.body.reason || "No reason specified",
            metadata: {
              method: req.method,
              endpoint: req.originalUrl,
              ip: req.ip,
              userAgent: req.headers["user-agent"],
              timestamp: new Date(),
            },
          });

          console.log(
            `✅ Logged moderation action: ${action} by ${req.user.username}`
          );
        }
      } catch (err) {
        console.error("❌ Moderation logging failed:", err.message);
      }
    });

    next();
  };
}

module.exports = moderationLogger;
