const express = require("express");
const router = express.Router();
const fs = require("fs");
const path = require("path");
const asyncHandler = require("express-async-handler");
const mongoose = require("mongoose");
const authValidator = require("../middleware/authValidator");
const { isBlogger } = require("../middleware/bloggerAuth");
const { uploadImage } = require("../helper/upload");
const moderationLogger = require("../middleware/moderationLogger");
const Story = require("../model/Story");
const StoryRating = require("../model/Storyrating");

// Helper function to generate slug from title
const generateSlug = (title) => {
  return title
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, "")
    .replace(/[\s_-]+/g, "-")
    .replace(/^-+|-+$/g, "");
};

// CREATE Story (with image upload) - Only Blogger/Admin
router.post(
  "/create",
  authValidator(),
  isBlogger,
  uploadImage.single("coverImage"),
  asyncHandler(async (req, res) => {
    const { title, excerpt, content, genres, tags, status } = req.body;

    // Debugging line (optional, helps verify request body)
    console.log("Incoming story body :", req.body);

    // VALIDATION
    if (!title || typeof title !== "string" || !title.trim()) {
      res.status(400);
      throw new Error("Story title is required");
    }

    if (!excerpt || typeof excerpt !== "string" || !excerpt.trim()) {
      res.status(400);
      throw new Error("Story excerpt is required");
    }

    if (!content || typeof content !== "string" || !content.trim()) {
      res.status(400);
      throw new Error("Story content is required");
    }

    // SLUG HANDLING
    let slug = generateSlug(title);
    const existingSlug = await Story.findOne({ slug });
    if (existingSlug) slug = `${slug}-${Date.now()}`;

    // GENRES
    let parsedGenres = [];
    if (genres) {
      try {
        parsedGenres = typeof genres === "string" ? JSON.parse(genres) : genres;

        // Ensure ObjectId type
        parsedGenres = parsedGenres.map(
          (id) => new mongoose.Types.ObjectId(id)
        );
      } catch (err) {
        console.error("Invalid genres format:", genres);
        res.status(400);
        throw new Error("Invalid genres data");
      }
    }

    // TAGS
    let parsedTags = [];
    if (tags) {
      try {
        parsedTags = typeof tags === "string" ? JSON.parse(tags) : tags;
        parsedTags = parsedTags.map((tag) => tag.trim());
      } catch (err) {
        console.error("Invalid tags format:", tags);
        res.status(400);
        throw new Error("Invalid tags data");
      }
    }

    // COVER IMAGE
    const coverImage = req.file ? `/uploads/images/${req.file.filename}` : null;

    //CREATE STORY
    const story = await Story.create({
      title: title.trim(),
      slug,
      excerpt: excerpt.trim(),
      content: content.trim(),
      coverImage,
      author: req.user.id,
      genres: parsedGenres,
      tags: parsedTags,
      status: status || "draft", // respects enum: ['draft','published','archived']
      views: 0,
      likesCount: 0,
      dislikesCount: 0,
      ratingsCount: 0,
      ratingsAvg: 0,
      moderation: {
        flagged: false,
        flagCount: 0,
      },
      isDeleted: false,
    });

    //POPULATE RESPONSE
    const populatedStory = await Story.findById(story.id)
      .populate("author", "username email firstName lastName role")
      .populate("genres", "name");

    res.status(200).json({
      success: true,
      message: "Story created successfully ",
      data: populatedStory,
    });
  })
);

// READ All Stories (with filters and pagination) - Public
router.get(
  "/fetch",
  asyncHandler(async (req, res) => {
    const { page = 1, limit = 10, author, genre, search, status } = req.query;

    const query = {
      isDeleted: false, // Exclude deleted stories by default
    };

    if (author && mongoose.Types.ObjectId.isValid(author))
      query.author = author;
    if (genre && mongoose.Types.ObjectId.isValid(genre)) query.genres = genre;
    if (status && ["draft", "published", "archived"].includes(status))
      query.status = status;

    if (search) {
      query.$or = [
        { title: { $regex: search, $options: "i" } },
        { excerpt: { $regex: search, $options: "i" } },
        { tags: { $regex: search, $options: "i" } },
      ];
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const [stories, total] = await Promise.all([
      Story.find(query)
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate("author", "username email firstName lastName pfp role")
        .populate("genres", "name"),
      Story.countDocuments(query),
    ]);

    res.status(200).json({
      success: true,
      count: stories.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      stories,
    });
  })
);

// READ Stroies by User ID - Public
router.get(
  "/fetch/user/:userId",
  asyncHandler(async (req, res) => {
    const { userId } = req.params;
    const { page = 1, limit = 10 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(userId)) {
      res.status(400);
      throw new Error("Invalid User ID");
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const query = {
      author: userId,
      isDeleted: false,
    };

    const [stories, total] = await Promise.all([
      Story.find(query)
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate(
          "author",
          "username email firstName lastName pfp occupation role"
        )
        .populate("genres", "name"),
      Story.countDocuments(query),
    ]);

    if (!stories || stories.length === 0) {
      res.status(404);
      throw new Error("No story found for this user");
    }

    res.status(200).json({
      success: true,
      count: stories.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      stories,
    });
  })
);

// READ One Story by ID - Public
router.get(
  "/fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Story ID");
    }

    const story = await Story.findOne({ _id: id, isDeleted: false })
      .populate(
        "author",
        "username email firstName lastName pfp occupation role"
      )
      .populate("genres", "name");

    if (!story) {
      res.status(404);
      throw new Error("Story not found");
    }

    // Increment view count
    story.views += 1;
    await story.save();

    res.status(200).json({ success: true, story });
  })
);

// READ One Story by Slug - Public
router.get(
  "/fetch/slug/:slug",
  asyncHandler(async (req, res) => {
    const { slug } = req.params;

    const story = await Story.findOne({ slug, isDeleted: false })
      .populate(
        "author",
        "username email firstName lastName pfp occupation role"
      )
      .populate("genres", "name");

    if (!story) {
      res.status(404);
      throw new Error("story not found");
    }

    // Increment view count
    story.views += 1;
    await story.save();

    res.status(200).json({ success: true, story });
  })
);

// UPDATE Story - Only Author or Admin
router.put(
  "/edit/:id",
  authValidator(),
  isBlogger,
  uploadImage.single("coverImage"),
  asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { title, excerpt, content, genres, tags, status } = req.body;

    // Validate ID
    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid story ID");
    }

    // Find story
    const story = await Story.findById(id);
    if (!story) {
      res.status(404);
      throw new Error("story not found");
    }

    // Authorization check (only author or admin can edit)
    if (
      req.user.role !== "admin" &&
      story.author.toString() !== req.user.id.toString()
    ) {
      res.status(403);
      throw new Error("Unauthorized: You can only edit your own stories");
    }

    // Prepare update data
    const updateData = {};

    // Update title + slug
    if (title && title.trim()) {
      const trimmedTitle = title.trim();
      const newSlug = generateSlug(trimmedTitle);

      // Check for existing slug (exclude current story)
      const slugExists = await Story.findOne({
        slug: newSlug,
        _id: { $ne: id },
      });

      updateData.title = trimmedTitle;
      updateData.slug = slugExists ? `${newSlug}-${Date.now()}` : newSlug;
    }

    if (excerpt) updateData.excerpt = excerpt.trim();
    if (content) updateData.content = content.trim();

    // Parse genres/tags if needed
    if (genres) {
      updateData.genres =
        typeof genres === "string" ? JSON.parse(genres) : genres;
    }
    if (tags) {
      updateData.tags = typeof tags === "string" ? JSON.parse(tags) : tags;
    }

    // Handle cover image update
    if (req.file) {
      // Delete old image if exists
      if (story.coverImage) {
        const oldImagePath = path.join(
          __dirname,
          "..",
          "views",
          story.coverImage
        );
        if (fs.existsSync(oldImagePath)) {
          fs.unlinkSync(oldImagePath);
        }
      }
      updateData.coverImage = `/uploads/images/${req.file.filename}`;
    }

    // Optional: update status (draft, published)
    if (status) updateData.status = status;

    // Update story
    const updatedStory = await Story.findByIdAndUpdate(id, updateData, {
      new: true,
      runValidators: true,
    })
      .populate("author", "username email firstName lastName role")
      .populate("genres", "name");

    res.status(200).json({
      success: true,
      message: "Story updated successfully",
      story: updatedStory,
    });
  })
);

// SOFT DELETE Story - Only Author or Admin
router.delete(
  "/delete/:id",
  authValidator(),
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid story ID");
    }

    const story = await Story.findById(id);
    if (!story) {
      res.status(404);
      throw new Error("Story not found");
    }

    // Mark as deleted
    story.isDeleted = true;
    story.status = "archived";
    await story.save();

    res.status(200).json({
      success: true,
      message: "Story soft deleted successfully",
    });
  })
);

// RESTORE SOFT DELETE Story - Only Author or Admin
router.put(
  "/restore/:id",
  authValidator(),
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Story ID");
    }

    const story = await Story.findById(id);
    if (!story) {
      res.status(404);
      throw new Error("Story not found");
    }

    // Only author or admin can restore
    if (
      story.author.toString() !== req.user.id.toString() &&
      req.user.role !== "admin"
    ) {
      res.status(403);
      throw new Error("Not authorized to restore this story");
    }

    if (!story.isDeleted) {
      res.status(400);
      throw new Error("Story is already active");
    }

    story.isDeleted = false;
    story.status = "draft"; // or "published" depending on your moderation logic
    await story.save();

    res.status(200).json({
      success: true,
      message: "Story restored successfully",
      story,
    });
  })
);

// HARD DELETE Story - Only Author or Admin
router.delete(
  "/:id/permanent",
  authValidator(),
  isBlogger,
  moderationLogger("DELETE_STORY", "story"),
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Story ID");
    }

    const story = await Story.findById(id);
    if (!story) {
      res.status(404);
      throw new Error("story not found");
    }

    // Delete permanently
    await Story.findByIdAndDelete(id);

    res.status(200).json({
      success: true,
      message: "Story permanently deleted",
    });
  })
);

// ADD or UPDATE STORY RATING
router.post(
  "/rate/:storyId",
  authValidator(),
  asyncHandler(async (req, res) => {
    const { storyId } = req.params;
    const { rating } = req.body;

    if (!mongoose.Types.ObjectId.isValid(storyId)) {
      res.status(400);
      throw new Error("Invalid Story ID");
    }

    if (!rating || rating < 1 || rating > 5) {
      res.status(400);
      throw new Error("Rating must be between 1 and 5");
    }

    const story = await Story.findById(storyId);
    if (!story) {
      res.status(404);
      throw new Error("Story not found");
    }

    // Check if user already rated
    const existingRating = await StoryRating.findOne({
      userId: req.user.id,
      storyId,
    });

    if (existingRating) {
      existingRating.rating = rating;
      await existingRating.save();
    } else {
      await StoryRating.create({
        userId: req.user.id,
        storyId,
        rating,
      });
    }

    // Recalculate story rating stats
    const ratings = await StoryRating.find({ storyId });
    const ratingsCount = ratings.length;
    const ratingsAvg =
      ratingsCount > 0
        ? ratings.reduce((acc, r) => acc + r.rating, 0) / ratingsCount
        : 0;

    story.ratingsCount = ratingsCount;
    story.ratingsAvg = ratingsAvg.toFixed(1);
    await story.save();

    res.status(200).json({
      success: true,
      message: "Rating submitted successfully",
      ratingsCount,
      ratingsAvg,
    });
  })
);

// GET STORY RATINGS SUMMARY
router.get(
  "/ratings/:storyId",
  asyncHandler(async (req, res) => {
    const { storyId } = req.params;

    if (!mongoose.Types.ObjectId.isValid(storyId)) {
      res.status(400);
      throw new Error("Invalid Story ID");
    }

    const story = await Story.findById(storyId)
      .select("ratingsAvg ratingsCount title")
      .lean();

    if (!story) {
      res.status(404);
      throw new Error("Story not found");
    }

    const allRatings = await StoryRating.find({ storyId })
      .populate("userId", "username email")
      .lean();

    res.status(200).json({
      success: true,
      story,    
      ratings: allRatings,
    });
  })
);

module.exports = router;
