const express = require("express");
const router = express.Router();
const asyncHandler = require("express-async-handler");
const mongoose = require("mongoose");
const Blog = require("../model/Blog");
const BlogRating = require("../model/Blograting");
const { uploadImage } = require("../helper/upload");
const { isBlogger, isAdmin } = require("../middleware/bloggerAuth");
const fs = require("fs");
const path = require("path");
const authValidator = require("../middleware/authValidator");
const moderationLogger = require("../middleware/moderationLogger");

// Helper function to generate slug from title
const generateSlug = (name) => {
  return name
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, "")
    .replace(/[\s_-]+/g, "-")
    .replace(/^-+|-+$/g, "");
};

  // CREATE Blog (with image upload) - Only Blogger/Admin
  router.post(
    "/create",
    authValidator(),
    isBlogger,
    uploadImage.single("coverImage"),
    asyncHandler(async (req, res) => {
      const { title, description, content, genres, tags, status } = req.body;

      // Debugging line (optional, helps verify request body)
      console.log("Incoming blog body:", req.body);

      // VALIDATION
      if (!title || typeof title !== "string" || !title.trim()) {
        res.status(400);
        throw new Error("Blog title is required");
      }

      if (
        !description ||
        typeof description !== "string" ||
        !description.trim()
      ) {
        res.status(400);
        throw new Error("Blog description is required");
      }

      if (!content || typeof content !== "string" || !content.trim()) {
        res.status(400);
        throw new Error("Blog content is required");
      }

      // SLUG HANDLING
      let slug = generateSlug(title);
      const existingSlug = await Blog.findOne({ slug });
      if (existingSlug) slug = `${slug}-${Date.now()}`;

      // GENRES
      let parsedGenres = [];
      if (genres) {
        try {
          parsedGenres = typeof genres === "string" ? JSON.parse(genres) : genres;

          // Ensure ObjectId type
          parsedGenres = parsedGenres.map(
            (id) => new mongoose.Types.ObjectId(id)
          );
        } catch (err) {
          console.error("Invalid genres format:", genres);
          res.status(400);
          throw new Error("Invalid genres data");
        }
      }

      // TAGS
      let parsedTags = [];
      if (tags) {
        try {
          parsedTags = typeof tags === "string" ? JSON.parse(tags) : tags;
          parsedTags = parsedTags.map((tag) => tag.trim());
        } catch (err) {
          console.error("Invalid tags format:", tags);
          res.status(400);
          throw new Error("Invalid tags data");
        }
      }

      // COVER IMAGE
      const coverImage = req.file ? `/uploads/images/${req.file.filename}` : null;

      // CREATE BLOG
      const blog = await Blog.create({
        title: title.trim(),
        slug,
        description: description.trim(),
        content: content.trim(),
        coverImage,
        author: req.user.id,
        genres: parsedGenres,
        tags: parsedTags,
        status: status || "draft", // respects enum: ['draft','published','archived']
        views: 0,
        likesCount: 0,
        dislikesCount: 0,
        ratingsCount: 0,
        ratingsAvg: 0,
        commentsCount: 0,
        moderation: {
          flagged: false,
          flagCount: 0,
        },
        isDeleted: false,
      });

      //POPULATE RESPONSE
      const populatedBlog = await Blog.findById(blog.id)
        .populate("author", "username email firstName lastName role")
        .populate("genres", "name");

      res.status(201).json({
        success: true,
        message: "Blog created successfully",
        data: populatedBlog,
      });
    })
  );

// READ All Blogs (with filters and pagination) - Public
router.get(
  "/fetch",
  asyncHandler(async (req, res) => {
    const { page = 1, limit = 10, author, genre, search, status } = req.query;

    const query = {
      isDeleted: false, // Exclude deleted blogs by default
    };

    if (author && mongoose.Types.ObjectId.isValid(author))
      query.author = author;
    if (genre && mongoose.Types.ObjectId.isValid(genre)) query.genres = genre;
    if (status && ["draft", "published", "archived"].includes(status))
      query.status = status;

    if (search) {
      query.$or = [
        { title: { $regex: search, $options: "i" } },
        { description: { $regex: search, $options: "i" } },
        { tags: { $regex: search, $options: "i" } },
      ];
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const [blogs, total] = await Promise.all([
      Blog.find(query)
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate("author", "username email firstName lastName pfp role")
        .populate("genres", "name"),
      Blog.countDocuments(query),
    ]);

    res.status(200).json({
      success: true,
      count: blogs.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      blogs,
    });
  })
);

// READ Blogs by User ID - Public
router.get(
  "/fetch/user/:userId",
  asyncHandler(async (req, res) => {
    const { userId } = req.params;
    const { page = 1, limit = 10 } = req.query;

    if (!mongoose.Types.ObjectId.isValid(userId)) {
      res.status(400);
      throw new Error("Invalid User ID");
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);

    const query = {
      author: userId,
      isDeleted: false,
    };

    const [blogs, total] = await Promise.all([
      Blog.find(query)
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(parseInt(limit))
        .populate(
          "author",
          "username email firstName lastName pfp occupation role"
        )
        .populate("genres", "name"),
      Blog.countDocuments(query),
    ]);

    if (!blogs || blogs.length === 0) {
      res.status(404);
      throw new Error("No blogs found for this user");
    }

    res.status(200).json({
      success: true,
      count: blogs.length,
      total,
      page: parseInt(page),
      totalPages: Math.ceil(total / parseInt(limit)),
      blogs,
    });
  })
);

// READ One Blog by ID - Public
router.get(
  "/fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findOne({ _id: id, isDeleted: false })
      .populate(
        "author",
        "username email firstName lastName pfp occupation role"
      )
      .populate("genres", "name");

    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Increment view count
    blog.views += 1;
    await blog.save();

    res.status(200).json({ success: true, blog });
  })
);

// READ One Blog by Slug - Public
router.get(
  "/fetch/slug/:slug",
  asyncHandler(async (req, res) => {
    const { slug } = req.params;

    const blog = await Blog.findOne({ slug, isDeleted: false })
      .populate(
        "author",
        "username email firstName lastName pfp occupation role"
      )
      .populate("genres", "name");

    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Increment view count
    blog.views += 1;
    await blog.save();

    res.status(200).json({ success: true, blog });
  })
);

  // UPDATE Blog - Only Author or Admin
  router.put(
    "/edit/:id",
    authValidator(),
    isBlogger,
    uploadImage.single("coverImage"),
    asyncHandler(async (req, res) => {
      const { id } = req.params;
      const { title, description, content, genres, tags, status } = req.body;

      // Validate ID
      if (!mongoose.Types.ObjectId.isValid(id)) {
        res.status(400);
        throw new Error("Invalid blog ID");
      }

      // Find blog
      const blog = await Blog.findById(id);
      if (!blog) {
        res.status(404);
        throw new Error("Blog not found");
      }

      // Authorization check (only author or admin can edit)
      if (
        req.user.role !== "admin" &&
        blog.author.toString() !== req.user.id.toString()
      ) {
        res.status(403);
        throw new Error("Unauthorized: You can only edit your own blogs");
      }

      // Prepare update data
      const updateData = {};

      // Update title + slug
      if (title && title.trim()) {
        const trimmedTitle = title.trim();
        const newSlug = generateSlug(trimmedTitle);

        // Check for existing slug (exclude current blog)
        const slugExists = await Blog.findOne({
          slug: newSlug,
          _id: { $ne: id },
        });

        updateData.title = trimmedTitle;
        updateData.slug = slugExists ? `${newSlug}-${Date.now()}` : newSlug;
      }

      if (description) updateData.description = description.trim();
      if (content) updateData.content = content.trim();

      // Parse genres/tags if needed
      if (genres) {
        updateData.genres =
          typeof genres === "string" ? JSON.parse(genres) : genres;
      }
      if (tags) {
        updateData.tags = typeof tags === "string" ? JSON.parse(tags) : tags;
      }

      // Handle cover image update
      if (req.file) {
        // Delete old image if exists
        if (blog.coverImage) {
          const oldImagePath = path.join(
            __dirname,
            "..",
            "views",
            blog.coverImage
          );
          if (fs.existsSync(oldImagePath)) {
            fs.unlinkSync(oldImagePath);
          }
        }
        updateData.coverImage = `/uploads/images/${req.file.filename}`;
      }

      // Optional: update status (draft, published)
      if (status) updateData.status = status;

      // Update blog
      const updatedBlog = await Blog.findByIdAndUpdate(id, updateData, {
        new: true,
        runValidators: true,
      })
        .populate("author", "username email firstName lastName role")
        .populate("genres", "name");

      res.status(200).json({
        success: true,
        message: "Blog updated successfully",
        blog: updatedBlog,
      });
    })
  );

// SOFT DELETE Blog - Only Author or Admin
router.delete(
  "/delete/:id",
  authValidator(),
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Mark as deleted
    blog.isDeleted = true;
    blog.status = "archived";
    await blog.save();

    res.status(200).json({
      success: true,
      message: "Blog soft deleted successfully",
    });
  })
);

// RESTORE SOFT DELETE Blog - Only Author or Admin
router.put(
  "/restore/:id",
  authValidator(),
  isBlogger,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Only author or admin can restore
    if (
      blog.author.toString() !== req.user.id.toString() &&
      req.user.role !== "admin"
    ) {
      res.status(403);
      throw new Error("Not authorized to restore this blog");
    }

    if (!blog.isDeleted) {
      res.status(400);
      throw new Error("Blog is already active");
    }

    blog.isDeleted = false;
    blog.status = "draft"; // or "published" depending on your moderation logic
    await blog.save();

    res.status(200).json({
      success: true,
      message: "Blog restored successfully",
      blog,
    });
  })
);

// RESTORE HARD DELETE Blog - Only Author or Admin
router.delete(
  "/:id/permanent",
  authValidator(),
  isBlogger,
  moderationLogger("DELETE_BLOG", "blog"),
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Delete permanently
    await Blog.findByIdAndDelete(id);

    res.status(200).json({
      success: true,
      message: "Blog permanently deleted",
    });
  })
);

// ADD or UPDATE BLOG RATING
router.post(
  "/rate/:blogId",
  authValidator(),
  asyncHandler(async (req, res) => {
    const { blogId } = req.params;
    const { rating } = req.body;

    if (!mongoose.Types.ObjectId.isValid(blogId)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    if (!rating || rating < 1 || rating > 5) {
      res.status(400);
      throw new Error("Rating must be between 1 and 5");
    }

    const blog = await Blog.findById(blogId);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Check if user already rated
    const existingRating = await BlogRating.findOne({
      userId: req.user.id,
      blogId,
    });

    if (existingRating) {
      existingRating.rating = rating;
      await existingRating.save();
    } else {
      await BlogRating.create({
        userId: req.user.id,
        blogId,
        rating,
      });
    }

    // Recalculate blog rating stats
    const ratings = await BlogRating.find({ blogId });
    const ratingsCount = ratings.length;
    const ratingsAvg =
      ratingsCount > 0
        ? ratings.reduce((acc, r) => acc + r.rating, 0) / ratingsCount
        : 0;

    blog.ratingsCount = ratingsCount;
    blog.ratingsAvg = ratingsAvg.toFixed(1);
    await blog.save();

    res.status(200).json({
      success: true,
      message: "Rating submitted successfully",
      ratingsCount,
      ratingsAvg,
    });
  })
);

// GET BLOG RATINGS SUMMARY
router.get(
  "/ratings/:blogId",
  asyncHandler(async (req, res) => {
    const { blogId } = req.params;

    if (!mongoose.Types.ObjectId.isValid(blogId)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(blogId)
      .select("ratingsAvg ratingsCount title")
      .lean();

    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    const allRatings = await BlogRating.find({ blogId })
      .populate("userId", "username email")
      .lean();

    res.status(200).json({
      success: true,
      blog,
      ratings: allRatings,
    });
  })
);

// LIKE/UNLIKE BLOG
router.post(
  "/:id/react",
  authValidator(),
  asyncHandler(async (req, res) => {
    const userId = req.user?.id;
    const { id } = req.params;
    const { action } = req.body;

    if (!["like", "dislike"].includes(action)) {
      res.status(400);
      throw new Error("Invalid action. Use 'like' or 'dislike'.");
    }

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Initialize arrays if they don't exist
    if (!blog.likes) blog.likes = [];
    if (!blog.dislikes) blog.dislikes = [];

    // Remove from both arrays first (ensures clean toggle)
    blog.likes = blog.likes.filter(
      (uid) => uid.toString() !== userId.toString()
    );
    blog.dislikes = blog.dislikes.filter(
      (uid) => uid.toString() !== userId.toString()
    );

    let message = "";
    if (action === "like") {
      blog.likes.push(userId);
      message = "Blog liked";
    } else if (action === "dislike") {
      blog.dislikes.push(userId);
      message = "Blog disliked";
    }

    await blog.save();

    res.status(200).json({
      success: true,
      message,
      likesCount: blog.likes.length,
      dislikesCount: blog.dislikes.length,
    });
  })
);

// FLAG Blog
router.post(
  "/:id/flag",
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const userId = req.user?.id;

    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Blog ID");
    }

    const blog = await Blog.findById(id);
    if (!blog) {
      res.status(404);
      throw new Error("Blog not found");
    }

    // Admin toggle clears all flags
    if (isAdmin) {
      const wasFlagged = blog.isFlagged;
      blog.flaggedBy = [];
      blog.isFlagged = false;
      await blog.save();

      return res.status(200).json({
        success: true,
        message: wasFlagged
          ? "All flags cleared by admin"
          : "No flags to clear",
        flaggedByCount: blog.flaggedBy.length,
      });
    }

    const alreadyFlagged = blog.flaggedBy.some(
      (uid) => uid.toString() === userId.toString()
    );

    if (alreadyFlagged) {
      blog.flaggedBy = blog.flaggedBy.filter(
        (uid) => uid.toString() !== userId.toString()
      );
      blog.isFlagged = blog.flaggedBy.length > 0;

      await blog.save();
      return res.status(200).json({
        success: true,
        message: "Flag removed successfully",
        flaggedByCount: blog.flaggedBy.length,
      });
    } else {
      blog.flaggedBy.push(userId);
      blog.isFlagged = true;

      await blog.save();
      return res.status(200).json({
        success: true,
        message: "Blog flagged successfully",
        flaggedByCount: blog.flaggedBy.length,
      });
    }
  })
);

module.exports = router;
